"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualDirectoryProvider = void 0;
const vscode = __importStar(require("vscode"));
/**
 * Provides virtual directory functionality for JAR files
 * Makes JAR files appear as expandable folders in the VS Code file explorer
 */
class VirtualDirectoryProvider {
    client;
    disposables = [];
    openedJars = new Set();
    // Handler configuration
    static HANDLER_ID = 'java-decompiler';
    static HANDLER_DISPLAY_NAME = 'Java Archive (JAR/WAR/EAR)';
    static FILE_PATTERNS = ['*.jar', '*.war', '*.ear'];
    constructor(client) {
        this.client = client;
    }
    /**
     * Register all commands required for virtual directory integration
     */
    register(context) {
        // List contents command
        this.disposables.push(vscode.commands.registerCommand('blackswan.virtualDirectory.listContents', (uri, relativePath) => this.listContents(uri, relativePath)));
        // Read file command
        this.disposables.push(vscode.commands.registerCommand('blackswan.virtualDirectory.readFile', (uri, relativePath) => this.readFile(uri, relativePath)));
        // On open command (called when a JAR is expanded)
        this.disposables.push(vscode.commands.registerCommand('blackswan.virtualDirectory.onOpen', (uri) => this.onOpen(uri)));
        // On close command (called when a JAR is collapsed/closed)
        this.disposables.push(vscode.commands.registerCommand('blackswan.virtualDirectory.onClose', (uri) => this.onClose(uri)));
        // Handler registration command - VS Code calls this to discover handlers
        this.disposables.push(vscode.commands.registerCommand('blackswan.virtualDirectory.register', () => this.getHandlerConfig()));
        // Legacy command names for backward compatibility
        this.disposables.push(vscode.commands.registerCommand('blackswan.archiveHandler.listContents', (uri, relativePath) => this.listContents(uri, relativePath)), vscode.commands.registerCommand('blackswan.archiveHandler.readFile', (uri, relativePath) => this.readFile(uri, relativePath)), vscode.commands.registerCommand('blackswan.archiveHandler.register', () => this.getHandlerConfig()));
        context.subscriptions.push(...this.disposables);
    }
    /**
     * Get the handler configuration for VS Code registration
     */
    getHandlerConfig() {
        return {
            id: VirtualDirectoryProvider.HANDLER_ID,
            displayName: VirtualDirectoryProvider.HANDLER_DISPLAY_NAME,
            filePatterns: VirtualDirectoryProvider.FILE_PATTERNS,
            listContentsCommand: 'blackswan.virtualDirectory.listContents',
            readFileCommand: 'blackswan.virtualDirectory.readFile',
            onOpenCommand: 'blackswan.virtualDirectory.onOpen',
            onCloseCommand: 'blackswan.virtualDirectory.onClose'
        };
    }
    /**
     * List contents of a JAR at a given relative path
     */
    async listContents(resourceUriStr, relativePath) {
        try {
            const jarPath = this.extractFilePath(resourceUriStr);
            console.log(`[VirtualDirectory] Listing contents of ${jarPath} at path: ${relativePath}`);
            // Ensure JAR is loaded
            await this.ensureJarLoaded(jarPath);
            // Get classes and files from the server
            const [classes, files] = await Promise.all([
                this.client.getClasses(),
                this.client.getFiles()
            ]);
            // Build entries for the requested path
            const entries = [];
            const seen = new Set();
            const normalizedRelPath = this.normalizePath(relativePath);
            const isRoot = normalizedRelPath === '';
            // Process classes (convert to .java file paths)
            for (const cls of classes) {
                const classPath = this.classToPath(cls.packageName, cls.simpleName);
                this.addEntryIfMatches(entries, seen, classPath, normalizedRelPath, isRoot);
            }
            // Process resource files
            for (const file of files) {
                this.addEntryIfMatches(entries, seen, file.name, normalizedRelPath, isRoot, file.size);
            }
            console.log(`[VirtualDirectory] Found ${entries.length} entries`);
            return entries;
        }
        catch (error) {
            console.error('[VirtualDirectory] Error listing contents:', error);
            return [];
        }
    }
    /**
     * Read a file from the JAR
     */
    async readFile(resourceUriStr, relativePath) {
        try {
            const jarPath = this.extractFilePath(resourceUriStr);
            const normalizedRelPath = this.normalizePath(relativePath);
            console.log(`[VirtualDirectory] Reading file ${normalizedRelPath} from ${jarPath}`);
            // Ensure JAR is loaded
            await this.ensureJarLoaded(jarPath);
            // Check if it's a .java file (needs decompilation)
            if (normalizedRelPath.endsWith('.java')) {
                return this.decompileClass(normalizedRelPath);
            }
            // Regular file - read from archive
            const content = await this.client.getFileContent(normalizedRelPath);
            return content;
        }
        catch (error) {
            console.error('[VirtualDirectory] Error reading file:', error);
            return `// Error reading file: ${error}`;
        }
    }
    /**
     * Called when a JAR is first opened/expanded
     */
    async onOpen(resourceUriStr) {
        const jarPath = this.extractFilePath(resourceUriStr);
        console.log(`[VirtualDirectory] Opening JAR: ${jarPath}`);
        await this.ensureJarLoaded(jarPath);
    }
    /**
     * Called when a JAR is closed/collapsed
     */
    async onClose(resourceUriStr) {
        const jarPath = this.extractFilePath(resourceUriStr);
        console.log(`[VirtualDirectory] Closing JAR: ${jarPath}`);
        // Could implement resource cleanup here if needed
    }
    /**
     * Ensure a JAR is loaded in the Recaf server
     */
    async ensureJarLoaded(jarPath) {
        if (!this.openedJars.has(jarPath)) {
            await this.client.openJar(jarPath);
            this.openedJars.add(jarPath);
        }
    }
    /**
     * Decompile a class file
     */
    async decompileClass(javaFilePath) {
        // Convert path to class name (com/example/Foo.java -> com/example/Foo)
        const className = javaFilePath.slice(0, -5); // Remove .java
        const result = await this.client.decompile(className);
        if (result.success && result.source) {
            return result.source;
        }
        else {
            return `// Decompilation failed: ${result.error || 'Unknown error'}`;
        }
    }
    /**
     * Convert class info to file path
     */
    classToPath(packageName, simpleName) {
        if (packageName) {
            // packageName might be in dot notation (com.example) or slash notation (com/example)
            const packagePath = packageName.replace(/\./g, '/');
            return `${packagePath}/${simpleName}.java`;
        }
        return `${simpleName}.java`;
    }
    /**
     * Add an entry to the list if it matches the current path
     */
    addEntryIfMatches(entries, seen, itemPath, normalizedRelPath, isRoot, size) {
        if (isRoot) {
            // At root level - get the first path component
            const firstPart = itemPath.split('/')[0];
            if (firstPart && !seen.has(firstPart)) {
                seen.add(firstPart);
                entries.push({
                    name: firstPart,
                    isDirectory: itemPath.split('/').length > 1,
                    size
                });
            }
        }
        else if (itemPath.startsWith(normalizedRelPath + '/')) {
            // Inside a subdirectory
            const remaining = itemPath.substring(normalizedRelPath.length + 1);
            const parts = remaining.split('/');
            if (parts.length > 0 && parts[0] && !seen.has(parts[0])) {
                seen.add(parts[0]);
                entries.push({
                    name: parts[0],
                    isDirectory: parts.length > 1,
                    size
                });
            }
        }
    }
    /**
     * Extract clean file path from URI string
     */
    extractFilePath(uriStr) {
        const uri = vscode.Uri.parse(uriStr);
        let filePath = uri.fsPath || uri.path;
        // Clean up path - remove URI artifacts
        filePath = filePath.replace(/^\/file:\/\/\//, '/');
        filePath = filePath.replace(/^\/file:\/\//, '/');
        filePath = filePath.replace(/^file:\/\//, '');
        filePath = filePath.replace(/^\/\/+/, '/');
        return filePath;
    }
    /**
     * Normalize a relative path
     */
    normalizePath(path) {
        // Remove leading slash and handle empty/root paths
        let normalized = path.startsWith('/') ? path.slice(1) : path;
        if (normalized === '/' || normalized === '.') {
            normalized = '';
        }
        return normalized;
    }
    dispose() {
        this.disposables.forEach(d => d.dispose());
        this.openedJars.clear();
    }
}
exports.VirtualDirectoryProvider = VirtualDirectoryProvider;
//# sourceMappingURL=VirtualDirectoryProvider.js.map