"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.JarFileSystemProvider = void 0;
const vscode = __importStar(require("vscode"));
const path = __importStar(require("path"));
/**
 * File system provider that makes JAR files appear as folders in VS Code.
 * Classes are shown as .java files that decompile on open.
 */
class JarFileSystemProvider {
    client;
    _onDidChangeFile;
    classes = new Map(); // jarPath -> classes
    files = new Map(); // jarPath -> files
    jarPaths = new Set(); // Track opened JARs
    loadingJars = new Set(); // Track JARs currently being loaded
    constructor(client) {
        this.client = client;
        this._onDidChangeFile = new vscode.EventEmitter();
    }
    get onDidChangeFile() {
        return this._onDidChangeFile.event;
    }
    watch(uri, options) {
        // No-op for now
        return new vscode.Disposable(() => { });
    }
    async stat(uri) {
        const jarPath = this.getJarPath(uri);
        const relativePath = this.getRelativePath(uri);
        console.log(`[JarFileSystemProvider] stat called: ${uri.toString()}`);
        console.log(`[JarFileSystemProvider] jarPath: ${jarPath}, relativePath: ${relativePath}`);
        // Root JAR URI - it's a directory
        if (!relativePath || relativePath === '') {
            return {
                type: vscode.FileType.Directory,
                ctime: Date.now(),
                mtime: Date.now(),
                size: 0
            };
        }
        // Ensure JAR is loaded before checking contents
        await this.ensureJarLoaded(jarPath);
        // Check if it's a class file (ends with .java in our virtual FS)
        if (relativePath.endsWith('.java')) {
            // Verify the class exists
            const classes = this.classes.get(jarPath) || [];
            const className = relativePath.slice(0, -5); // Remove .java
            const classExists = classes.some(c => {
                const classPath = c.packageName ? `${c.packageName}/${c.simpleName}` : c.simpleName;
                return classPath === className;
            });
            if (classExists) {
                return {
                    type: vscode.FileType.File,
                    ctime: Date.now(),
                    mtime: Date.now(),
                    size: 0
                };
            }
        }
        // Check if it's a directory (package path)
        const classes = this.classes.get(jarPath) || [];
        const files = this.files.get(jarPath) || [];
        // Check if any class or file is in this directory
        const isDirectory = classes.some(c => {
            const classPath = c.packageName ? `${c.packageName}/${c.simpleName}.java` : `${c.simpleName}.java`;
            return classPath.startsWith(relativePath + '/');
        }) || files.some(f => {
            return f.name.startsWith(relativePath + '/');
        });
        if (isDirectory) {
            return {
                type: vscode.FileType.Directory,
                ctime: Date.now(),
                mtime: Date.now(),
                size: 0
            };
        }
        // Check if it's a regular file
        const file = files.find(f => f.name === relativePath);
        if (file) {
            return {
                type: vscode.FileType.File,
                ctime: Date.now(),
                mtime: Date.now(),
                size: file.size
            };
        }
        console.log(`[JarFileSystemProvider] File not found: ${relativePath}`);
        throw vscode.FileSystemError.FileNotFound(uri);
    }
    async readDirectory(uri) {
        console.log(`[JarFileSystemProvider] readDirectory called: ${uri.toString()}`);
        console.log(`[JarFileSystemProvider] readDirectory scheme: ${uri.scheme}`);
        const jarPath = this.getJarPath(uri);
        const relativePath = this.getRelativePath(uri) || '';
        console.log(`[JarFileSystemProvider] readDirectory jarPath: ${jarPath}, relativePath: ${relativePath}`);
        // Check if JAR is currently being loaded - show loading indicator
        if (this.loadingJars.has(jarPath)) {
            console.log(`[JarFileSystemProvider] JAR is loading: ${jarPath}`);
            return [['⏳ Loading...', vscode.FileType.File]];
        }
        // Ensure JAR is loaded
        await this.ensureJarLoaded(jarPath);
        const classes = this.classes.get(jarPath) || [];
        const files = this.files.get(jarPath) || [];
        const entries = new Map();
        console.log(`[JarFileSystemProvider] Found ${classes.length} classes, ${files.length} files`);
        // Add classes
        for (const cls of classes) {
            const classPath = cls.packageName ? `${cls.packageName}/${cls.simpleName}.java` : `${cls.simpleName}.java`;
            if (relativePath === '') {
                // Root level - show first package/file
                const firstPart = classPath.split('/')[0];
                if (firstPart && !entries.has(firstPart)) {
                    // Check if it's a directory (has more parts) or file (ends with .java)
                    if (classPath.split('/').length > 1) {
                        entries.set(firstPart, vscode.FileType.Directory);
                    }
                    else {
                        entries.set(firstPart, vscode.FileType.File);
                    }
                }
            }
            else if (classPath.startsWith(relativePath + '/')) {
                // Inside a directory - show next level
                const remaining = classPath.substring(relativePath.length + 1);
                const parts = remaining.split('/');
                if (parts.length > 0 && parts[0]) {
                    if (parts.length > 1) {
                        // It's a subdirectory
                        entries.set(parts[0], vscode.FileType.Directory);
                    }
                    else {
                        // It's a file
                        entries.set(parts[0], vscode.FileType.File);
                    }
                }
            }
        }
        // Add regular files
        for (const file of files) {
            if (relativePath === '') {
                const firstPart = file.name.split('/')[0];
                if (firstPart && !entries.has(firstPart)) {
                    // Check if it's a directory or file
                    if (file.name.split('/').length > 1) {
                        entries.set(firstPart, vscode.FileType.Directory);
                    }
                    else {
                        entries.set(firstPart, vscode.FileType.File);
                    }
                }
            }
            else if (file.name.startsWith(relativePath + '/')) {
                const remaining = file.name.substring(relativePath.length + 1);
                const parts = remaining.split('/');
                if (parts.length > 0 && parts[0]) {
                    if (parts.length > 1) {
                        entries.set(parts[0], vscode.FileType.Directory);
                    }
                    else {
                        entries.set(parts[0], vscode.FileType.File);
                    }
                }
            }
        }
        return Array.from(entries.entries());
    }
    async readFile(uri) {
        console.log(`[JarFileSystemProvider] readFile called: ${uri.toString()}`);
        console.log(`[JarFileSystemProvider] readFile scheme: ${uri.scheme}`);
        const jarPath = this.getJarPath(uri);
        const relativePath = this.getRelativePath(uri);
        console.log(`[JarFileSystemProvider] readFile jarPath: ${jarPath}, relativePath: ${relativePath}`);
        if (!relativePath) {
            throw vscode.FileSystemError.FileNotFound(uri);
        }
        // Ensure JAR is loaded
        await this.ensureJarLoaded(jarPath);
        // If it's a .java file, decompile it
        if (relativePath.endsWith('.java')) {
            const className = relativePath.slice(0, -5); // Remove .java
            console.log(`[JarFileSystemProvider] Decompiling class: ${className}`);
            try {
                const result = await this.client.decompile(className);
                if (result.success && result.source) {
                    return new TextEncoder().encode(result.source);
                }
                else {
                    const errorMsg = `// Decompilation failed: ${result.error || 'Unknown error'}`;
                    return new TextEncoder().encode(errorMsg);
                }
            }
            catch (error) {
                const errorMsg = `// Error decompiling class: ${error}`;
                return new TextEncoder().encode(errorMsg);
            }
        }
        // Regular file - get content from server
        const files = this.files.get(jarPath) || [];
        const file = files.find(f => f.name === relativePath);
        if (file) {
            try {
                const content = await this.client.getFileContent(file.name);
                return new TextEncoder().encode(content);
            }
            catch (error) {
                const errorMsg = `// Error reading file: ${error}`;
                return new TextEncoder().encode(errorMsg);
            }
        }
        throw vscode.FileSystemError.FileNotFound(uri);
    }
    writeFile(uri, content, options) {
        // Read-only for now
        throw vscode.FileSystemError.NoPermissions(uri);
    }
    delete(uri, options) {
        throw vscode.FileSystemError.NoPermissions(uri);
    }
    rename(oldUri, newUri, options) {
        throw vscode.FileSystemError.NoPermissions(oldUri);
    }
    createDirectory(uri) {
        throw vscode.FileSystemError.NoPermissions(uri);
    }
    /**
     * Extract JAR path from URI (jar:file:///path/to/file.jar/...)
     */
    getJarPath(uri) {
        // URI format: jar:file:///absolute/path/to/file.jar/path/inside/jar
        // For Windows: jar:file:///C:/path/to/file.jar/...
        let pathStr = uri.path;
        console.log(`[JarFileSystemProvider] getJarPath input: ${pathStr}`);
        // Clean up URI artifacts first - handle all possible formats
        // Order matters: more specific patterns first
        pathStr = pathStr.replace(/^\/\/file:\/\/\//, '/'); // Remove //file:///
        pathStr = pathStr.replace(/^\/\/file:\/\//, '/'); // Remove //file://
        pathStr = pathStr.replace(/^\/file:\/\/\//, '/'); // Remove /file:/// (single leading slash)
        pathStr = pathStr.replace(/^\/file:\/\//, '/'); // Remove /file:// (single leading slash)
        pathStr = pathStr.replace(/^\/file:/, ''); // Remove /file: (no slashes after colon)
        pathStr = pathStr.replace(/^file:\/\/\//, '/'); // Remove file:///
        pathStr = pathStr.replace(/^file:\/\//, ''); // Remove file://
        pathStr = pathStr.replace(/^file:/, ''); // Remove file: (no slashes)
        pathStr = pathStr.replace(/^\/\/+/, '/'); // Remove multiple leading slashes
        // Handle Windows paths (C:/...)
        if (pathStr.match(/^\/[A-Z]:/i)) {
            pathStr = pathStr.substring(1); // Remove leading slash for Windows
        }
        // Ensure leading slash for absolute paths on Unix
        if (!pathStr.startsWith('/') && !pathStr.match(/^[A-Z]:/i)) {
            pathStr = '/' + pathStr;
        }
        console.log(`[JarFileSystemProvider] getJarPath after cleanup: ${pathStr}`);
        const pathParts = pathStr.split('/');
        // Find the .jar file in the path
        for (let i = 0; i < pathParts.length; i++) {
            if (pathParts[i].endsWith('.jar')) {
                let jarPath = pathParts.slice(0, i + 1).join('/');
                // Ensure absolute path (add leading slash for Unix)
                if (!jarPath.startsWith('/') && !jarPath.match(/^[A-Z]:/i)) {
                    jarPath = '/' + jarPath;
                }
                // Clean up any remaining URI artifacts
                jarPath = jarPath.replace(/^\/\/+/, '/');
                console.log(`[JarFileSystemProvider] getJarPath result: ${jarPath}`);
                return jarPath;
            }
        }
        // If no .jar found, clean up and return the path
        pathStr = pathStr.replace(/^\/\/+/, '/');
        console.log(`[JarFileSystemProvider] getJarPath result (no jar found): ${pathStr}`);
        return pathStr;
    }
    /**
     * Get relative path inside JAR (everything after .jar/)
     */
    getRelativePath(uri) {
        let pathStr = uri.path;
        // Clean up URI artifacts first - same patterns as getJarPath
        pathStr = pathStr.replace(/^\/\/file:\/\/\//, '/');
        pathStr = pathStr.replace(/^\/\/file:\/\//, '/');
        pathStr = pathStr.replace(/^\/file:\/\/\//, '/');
        pathStr = pathStr.replace(/^\/file:\/\//, '/');
        pathStr = pathStr.replace(/^\/file:/, ''); // Handle /file: (no slashes after colon)
        pathStr = pathStr.replace(/^file:\/\/\//, '/');
        pathStr = pathStr.replace(/^file:\/\//, '');
        pathStr = pathStr.replace(/^file:/, ''); // Handle file: (no slashes)
        pathStr = pathStr.replace(/^\/\/+/, '/');
        // Handle Windows paths
        if (pathStr.match(/^\/[A-Z]:/i)) {
            pathStr = pathStr.substring(1);
        }
        // Ensure leading slash for absolute paths on Unix
        if (!pathStr.startsWith('/') && !pathStr.match(/^[A-Z]:/i)) {
            pathStr = '/' + pathStr;
        }
        const pathParts = pathStr.split('/');
        for (let i = 0; i < pathParts.length; i++) {
            if (pathParts[i].endsWith('.jar')) {
                return pathParts.slice(i + 1).join('/');
            }
        }
        return '';
    }
    /**
     * Ensure JAR is loaded in the workspace
     */
    async ensureJarLoaded(jarPath) {
        // Clean up the path one more time to ensure no URI artifacts
        // Handle all possible formats - same patterns as getJarPath
        jarPath = jarPath.replace(/^\/\/file:\/\/\//, '/');
        jarPath = jarPath.replace(/^\/\/file:\/\//, '/');
        jarPath = jarPath.replace(/^\/file:\/\/\//, '/'); // /file:/// -> /
        jarPath = jarPath.replace(/^\/file:\/\//, '/'); // /file:// -> /
        jarPath = jarPath.replace(/^\/file:/, ''); // /file: -> empty (no slashes after colon)
        jarPath = jarPath.replace(/^file:\/\/\//, '/'); // file:/// -> /
        jarPath = jarPath.replace(/^file:\/\//, ''); // file:// -> empty
        jarPath = jarPath.replace(/^file:/, ''); // file: -> empty (no slashes)
        jarPath = jarPath.replace(/^\/\/+/, '/');
        // Ensure leading slash for absolute paths on Unix
        if (!jarPath.startsWith('/') && !jarPath.match(/^[A-Z]:/i)) {
            jarPath = '/' + jarPath;
        }
        if (this.jarPaths.has(jarPath)) {
            return; // Already loaded
        }
        // If already loading, wait for it
        if (this.loadingJars.has(jarPath)) {
            console.log(`[JarFileSystemProvider] JAR already loading, waiting: ${jarPath}`);
            // Wait a bit and retry
            await new Promise(resolve => setTimeout(resolve, 100));
            return this.ensureJarLoaded(jarPath);
        }
        // Mark as loading
        this.loadingJars.add(jarPath);
        console.log(`[JarFileSystemProvider] Loading JAR: ${jarPath}`);
        try {
            // Open the JAR in the workspace
            await this.client.openJar(jarPath);
            // Load classes and files
            const [classes, files] = await Promise.all([
                this.client.getClasses(),
                this.client.getFiles()
            ]);
            console.log(`[JarFileSystemProvider] Loaded ${classes.length} classes, ${files.length} files from ${jarPath}`);
            this.classes.set(jarPath, classes);
            this.files.set(jarPath, files);
            this.jarPaths.add(jarPath);
            // Fire change event to refresh the tree
            const jarUri = vscode.Uri.from({ scheme: 'jar', path: jarPath });
            this._onDidChangeFile.fire([{ type: vscode.FileChangeType.Changed, uri: jarUri }]);
        }
        catch (error) {
            console.error(`Failed to load JAR ${jarPath}:`, error);
            throw vscode.FileSystemError.Unavailable(`Failed to load JAR: ${error}`);
        }
        finally {
            // Always remove from loading set
            this.loadingJars.delete(jarPath);
        }
    }
    /**
     * Create a URI for a JAR file
     */
    static createJarUri(jarPath, relativePath = '') {
        // Normalize the path
        let normalizedPath = path.normalize(jarPath).replace(/\\/g, '/');
        // Ensure absolute path
        if (!path.isAbsolute(jarPath)) {
            normalizedPath = '/' + normalizedPath;
        }
        // Handle Windows paths (C:\... -> /C:/...)
        if (normalizedPath.match(/^[A-Z]:/i)) {
            normalizedPath = '/' + normalizedPath;
        }
        const uriPath = normalizedPath + (relativePath ? '/' + relativePath : '');
        return vscode.Uri.parse(`jar:file://${uriPath}`);
    }
}
exports.JarFileSystemProvider = JarFileSystemProvider;
//# sourceMappingURL=JarFileSystemProvider.js.map