"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClassTreeProvider = exports.ClassTreeItem = void 0;
const vscode = __importStar(require("vscode"));
class ClassTreeItem extends vscode.TreeItem {
    label;
    collapsibleState;
    classInfo;
    memberInfo;
    parentClass;
    packagePath;
    constructor(label, collapsibleState, classInfo, memberInfo, parentClass, packagePath) {
        super(label, collapsibleState);
        this.label = label;
        this.collapsibleState = collapsibleState;
        this.classInfo = classInfo;
        this.memberInfo = memberInfo;
        this.parentClass = parentClass;
        this.packagePath = packagePath;
        if (classInfo) {
            // This is a class node
            this.contextValue = 'class';
            this.iconPath = this.getClassIcon(classInfo.access);
            this.tooltip = this.getClassTooltip(classInfo);
            this.description = `v${classInfo.version - 44}`; // Convert to Java version
            // Command to decompile on click
            this.command = {
                command: 'recaf.decompileClass',
                title: 'Decompile',
                arguments: [classInfo.name]
            };
        }
        else if (memberInfo) {
            // This is a member node (field or method)
            const isMethod = memberInfo.descriptor.startsWith('(');
            this.contextValue = isMethod ? 'method' : 'field';
            this.iconPath = isMethod
                ? new vscode.ThemeIcon('symbol-method')
                : new vscode.ThemeIcon('symbol-field');
            this.tooltip = `${memberInfo.name}${memberInfo.descriptor}`;
            this.description = this.getShortDescriptor(memberInfo.descriptor);
        }
        else {
            // This is a package node
            this.contextValue = 'package';
            this.iconPath = new vscode.ThemeIcon('symbol-package');
        }
    }
    getClassIcon(access) {
        // Access flags
        const ACC_INTERFACE = 0x0200;
        const ACC_ABSTRACT = 0x0400;
        const ACC_ANNOTATION = 0x2000;
        const ACC_ENUM = 0x4000;
        if (access & ACC_ANNOTATION) {
            return new vscode.ThemeIcon('symbol-interface');
        }
        if (access & ACC_INTERFACE) {
            return new vscode.ThemeIcon('symbol-interface');
        }
        if (access & ACC_ENUM) {
            return new vscode.ThemeIcon('symbol-enum');
        }
        if (access & ACC_ABSTRACT) {
            return new vscode.ThemeIcon('symbol-class');
        }
        return new vscode.ThemeIcon('symbol-class');
    }
    getClassTooltip(info) {
        const lines = [
            `Class: ${info.name}`,
            `Super: ${info.superName}`,
        ];
        if (info.interfaces.length > 0) {
            lines.push(`Interfaces: ${info.interfaces.join(', ')}`);
        }
        lines.push(`Version: ${info.version} (Java ${info.version - 44})`);
        return lines.join('\n');
    }
    getShortDescriptor(desc) {
        // Simplify descriptors for display
        return desc
            .replace(/Ljava\/lang\/String;/g, 'String')
            .replace(/Ljava\/lang\/Object;/g, 'Object')
            .replace(/L([^;]+);/g, (_, name) => name.split('/').pop() || name)
            .replace(/\[/g, '[]');
    }
}
exports.ClassTreeItem = ClassTreeItem;
class ClassTreeProvider {
    client;
    _onDidChangeTreeData = new vscode.EventEmitter();
    onDidChangeTreeData = this._onDidChangeTreeData.event;
    classes = [];
    packageTree = null;
    constructor(client) {
        this.client = client;
    }
    refresh() {
        this.classes = [];
        this.packageTree = null;
        this._onDidChangeTreeData.fire();
    }
    async loadClasses() {
        try {
            // Show loading indicator for large JARs
            const classes = await Promise.race([
                this.client.getClasses(),
                new Promise((_, reject) => setTimeout(() => reject(new Error('Request timeout')), 30000))
            ]);
            this.classes = classes;
            this.packageTree = this.buildPackageTree(this.classes);
            this._onDidChangeTreeData.fire();
        }
        catch (error) {
            console.error('Failed to load classes:', error);
            vscode.window.showErrorMessage(`Failed to load classes: ${error}`);
        }
    }
    buildPackageTree(classes) {
        const root = {
            name: '',
            classes: [],
            children: new Map()
        };
        for (const cls of classes) {
            const parts = cls.packageName.split('/').filter(p => p.length > 0);
            let current = root;
            for (const part of parts) {
                if (!current.children.has(part)) {
                    current.children.set(part, {
                        name: part,
                        classes: [],
                        children: new Map()
                    });
                }
                current = current.children.get(part);
            }
            current.classes.push(cls);
        }
        return root;
    }
    getTreeItem(element) {
        return element;
    }
    async getChildren(element) {
        if (!element) {
            // Root level - load classes if needed
            if (this.classes.length === 0) {
                try {
                    await this.loadClasses();
                }
                catch {
                    return [];
                }
            }
            if (!this.packageTree) {
                return [];
            }
            // Return root packages and classes
            return this.getPackageChildren(this.packageTree);
        }
        if (element.contextValue === 'package') {
            // Get children of a package
            if (!this.packageTree)
                return [];
            const packagePath = element.packagePath || element.label;
            const node = this.findPackageNode(this.packageTree, packagePath);
            if (node) {
                return this.getPackageChildren(node, packagePath);
            }
            return [];
        }
        if (element.contextValue === 'class' && element.classInfo) {
            // Get members of a class
            try {
                const members = await this.client.getMembers(element.classInfo.name);
                const items = [];
                // Fields
                for (const field of members.fields) {
                    items.push(new ClassTreeItem(field.name, vscode.TreeItemCollapsibleState.None, undefined, field, element.classInfo.name));
                }
                // Methods
                for (const method of members.methods) {
                    items.push(new ClassTreeItem(method.name, vscode.TreeItemCollapsibleState.None, undefined, method, element.classInfo.name));
                }
                return items;
            }
            catch (error) {
                console.error('Failed to get members:', error);
                return [];
            }
        }
        return [];
    }
    getPackageChildren(node, parentPath = '') {
        const items = [];
        // Add child packages
        for (const [name, child] of node.children) {
            const fullPath = parentPath ? `${parentPath}/${name}` : name;
            items.push(new ClassTreeItem(name, vscode.TreeItemCollapsibleState.Collapsed, undefined, undefined, undefined, fullPath));
        }
        // Add classes in this package
        for (const cls of node.classes) {
            items.push(new ClassTreeItem(cls.simpleName, vscode.TreeItemCollapsibleState.Collapsed, cls));
        }
        return items;
    }
    findPackageNode(root, fullPath) {
        const parts = fullPath.split('/').filter(p => p.length > 0);
        let current = root;
        for (const part of parts) {
            if (current.children.has(part)) {
                current = current.children.get(part);
            }
            else {
                return null;
            }
        }
        return current;
    }
    getParent(element) {
        // Not implemented for now
        return null;
    }
}
exports.ClassTreeProvider = ClassTreeProvider;
//# sourceMappingURL=ClassTreeProvider.js.map