"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.activate = activate;
exports.deactivate = deactivate;
exports.getClient = getClient;
const vscode = __importStar(require("vscode"));
const RecafClient_1 = require("./client/RecafClient");
const ClassTreeProvider_1 = require("./providers/ClassTreeProvider");
const FileTreeProvider_1 = require("./providers/FileTreeProvider");
const MappingsTreeProvider_1 = require("./providers/MappingsTreeProvider");
const SearchResultsProvider_1 = require("./providers/SearchResultsProvider");
const DecompiledDocumentProvider_1 = require("./providers/DecompiledDocumentProvider");
const VirtualDirectoryProvider_1 = require("./providers/VirtualDirectoryProvider");
const commands_1 = require("./commands");
let client;
async function activate(context) {
    console.log('Java Decompiler extension activating...');
    // Initialize the Java decompiler client
    client = new RecafClient_1.RecafClient(context);
    // Register the virtual directory provider (for native explorer integration)
    // This allows JAR files to appear as expandable folders inline in the file explorer
    const virtualDirectoryProvider = new VirtualDirectoryProvider_1.VirtualDirectoryProvider(client);
    virtualDirectoryProvider.register(context);
    context.subscriptions.push(virtualDirectoryProvider);
    // Register vdir: scheme file system provider for reading virtual directory contents
    const vdirFsProvider = new VdirFileSystemProvider(virtualDirectoryProvider);
    context.subscriptions.push(vscode.workspace.registerFileSystemProvider('vdir', vdirFsProvider, { isCaseSensitive: false, isReadonly: true }));
    // Intercept attempts to open files inside JAR as regular file:// URIs
    // This happens when VS Code incorrectly resolves paths through the filesystem
    const jarOpenMessageShown = new Set();
    context.subscriptions.push(vscode.workspace.onDidOpenTextDocument(async (document) => {
        const uri = document.uri;
        // Check if it's trying to access a path inside a JAR file using file:// scheme
        if (uri.scheme === 'file') {
            const fsPath = uri.fsPath;
            // Check if path contains .jar/ - indicates trying to access inside a JAR
            const jarMatch = fsPath.match(/(.+\.jar)[\/\\](.+)/);
            if (jarMatch) {
                const jarPath = jarMatch[1];
                const relativePath = jarMatch[2].replace(/\\/g, '/');
                console.log(`[Extension] Intercepted file:// access to JAR contents: ${fsPath}`);
                console.log(`[Extension] JAR: ${jarPath}, Relative: ${relativePath}`);
                // Close the failed document
                await vscode.commands.executeCommand('workbench.action.closeActiveEditor');
                // Create vdir: URI (the correct scheme for virtual directory contents)
                const jarUri = vscode.Uri.file(jarPath);
                const vdirUri = createVdirUri(jarUri, relativePath);
                console.log(`[Extension] Opening with vdir URI: ${vdirUri.toString()}`);
                await vscode.window.showTextDocument(vdirUri);
                return;
            }
            // Check if it's a JAR file being opened as binary - provide helpful guidance
            if (fsPath.endsWith('.jar')) {
                // Only show message once per JAR file per session
                if (!jarOpenMessageShown.has(fsPath)) {
                    jarOpenMessageShown.add(fsPath);
                    vscode.window.showInformationMessage('Expand the JAR file in the file explorer to browse its contents.', 'Got it');
                }
            }
        }
    }));
    // Register the virtual document provider for decompiled sources (for backward compatibility)
    const decompiledProvider = new DecompiledDocumentProvider_1.DecompiledDocumentProvider(client);
    context.subscriptions.push(vscode.workspace.registerTextDocumentContentProvider('recaf', decompiledProvider));
    // Register tree view providers
    const classTreeProvider = new ClassTreeProvider_1.ClassTreeProvider(client);
    const fileTreeProvider = new FileTreeProvider_1.FileTreeProvider(client);
    const mappingsTreeProvider = new MappingsTreeProvider_1.MappingsTreeProvider(client);
    const searchResultsProvider = new SearchResultsProvider_1.SearchResultsProvider(client);
    context.subscriptions.push(vscode.window.createTreeView('recaf.classes', {
        treeDataProvider: classTreeProvider,
        showCollapseAll: true
    }), vscode.window.createTreeView('recaf.files', {
        treeDataProvider: fileTreeProvider,
        showCollapseAll: true
    }), vscode.window.createTreeView('recaf.mappings', {
        treeDataProvider: mappingsTreeProvider,
        showCollapseAll: true
    }), vscode.window.createTreeView('recaf.searchResults', {
        treeDataProvider: searchResultsProvider,
        showCollapseAll: true
    }));
    // Register commands (must be done before server start so commands are available)
    (0, commands_1.registerCommands)(context, client, {
        classTree: classTreeProvider,
        fileTree: fileTreeProvider,
        mappingsTree: mappingsTreeProvider,
        searchResults: searchResultsProvider,
        decompiledProvider
    });
    // Try to start the server (non-blocking - commands are already registered)
    try {
        await client.start();
        console.log('Java decompiler server started successfully');
        vscode.window.showInformationMessage('Java Decompiler is ready! Use "Open JAR File" to get started.');
    }
    catch (error) {
        console.error('Failed to start Java decompiler server:', error);
        vscode.window.showWarningMessage(`Java Decompiler: Server not available. Some features may be limited. ${error}`);
    }
    context.subscriptions.push({
        dispose: () => {
            if (client) {
                client.stop();
            }
        }
    });
}
/**
 * Extract clean file path from URI
 */
function extractFilePath(uri) {
    let filePath = uri.fsPath || uri.path;
    // If fsPath doesn't work or gives us a URI-like path, try to clean it
    if (!filePath || filePath.includes('://') || filePath.startsWith('//file:')) {
        filePath = uri.path;
    }
    // Clean up path - remove URI artifacts
    filePath = filePath.replace(/^\/\/file:\/\/\//, '/');
    filePath = filePath.replace(/^\/\/file:\/\//, '/');
    filePath = filePath.replace(/^\/file:\/\/\//, '/');
    filePath = filePath.replace(/^\/file:\/\//, '/');
    filePath = filePath.replace(/^file:\/\//, '');
    filePath = filePath.replace(/^\/\/+/, '/');
    return filePath;
}
function deactivate() {
    if (client) {
        client.stop();
    }
}
function getClient() {
    return client;
}
/**
 * Create a vdir: URI for content inside a virtual directory (e.g., JAR file)
 * Format: vdir:///<base64url-encoded-archive-uri>/<relative-path>
 */
function createVdirUri(archiveUri, relativePath) {
    // Encode using base64, then convert to base64url (replace + with -, / with _, remove =)
    const base64 = Buffer.from(archiveUri.toString()).toString('base64');
    const archiveUriEncoded = base64.replace(/\+/g, '-').replace(/\//g, '_').replace(/=/g, '');
    return vscode.Uri.from({
        scheme: 'vdir',
        path: `/${archiveUriEncoded}/${relativePath}`
    });
}
/**
 * Parse a vdir: URI to extract the archive URI and relative path
 * Format: vdir:///<base64url-encoded-archive-uri>/<relative-path>
 */
function parseVdirUri(uri) {
    if (uri.scheme !== 'vdir') {
        return undefined;
    }
    const parts = uri.path.split('/');
    if (parts.length < 2) {
        return undefined;
    }
    // First part is empty (leading /), second is the base64 encoded archive URI
    const archiveUriEncoded = parts[1];
    const relativePath = parts.slice(2).join('/');
    try {
        // Decode base64url (replace - with +, _ with /)
        const base64Standard = archiveUriEncoded.replace(/-/g, '+').replace(/_/g, '/');
        const archiveUriStr = Buffer.from(base64Standard, 'base64').toString('utf8');
        const archiveUri = vscode.Uri.parse(archiveUriStr);
        return { archiveUri, relativePath };
    }
    catch {
        return undefined;
    }
}
/**
 * File system provider for vdir: scheme URIs
 * Handles reading files from inside virtual directories (like JAR files)
 */
class VdirFileSystemProvider {
    virtualDirectoryProvider;
    _onDidChangeFile = new vscode.EventEmitter();
    onDidChangeFile = this._onDidChangeFile.event;
    constructor(virtualDirectoryProvider) {
        this.virtualDirectoryProvider = virtualDirectoryProvider;
    }
    watch() {
        return new vscode.Disposable(() => { });
    }
    async stat(uri) {
        const parsed = parseVdirUri(uri);
        if (!parsed) {
            throw vscode.FileSystemError.FileNotFound(uri);
        }
        // Determine if it's a file or directory based on the relative path
        // For now, assume paths ending with common extensions are files
        const isFile = /\.(java|class|xml|txt|properties|json|yml|yaml|md|html|css|js|ts)$/i.test(parsed.relativePath);
        return {
            type: isFile ? vscode.FileType.File : vscode.FileType.Directory,
            ctime: Date.now(),
            mtime: Date.now(),
            size: 0
        };
    }
    async readDirectory(uri) {
        const parsed = parseVdirUri(uri);
        if (!parsed) {
            return [];
        }
        const entries = await this.virtualDirectoryProvider.listContents(parsed.archiveUri.toString(), parsed.relativePath);
        return entries.map(entry => [
            entry.name,
            entry.isDirectory ? vscode.FileType.Directory : vscode.FileType.File
        ]);
    }
    async readFile(uri) {
        const parsed = parseVdirUri(uri);
        if (!parsed) {
            throw vscode.FileSystemError.FileNotFound(uri);
        }
        const content = await this.virtualDirectoryProvider.readFile(parsed.archiveUri.toString(), parsed.relativePath);
        if (typeof content === 'string') {
            return new TextEncoder().encode(content);
        }
        return content;
    }
    // Read-only operations
    writeFile() {
        throw vscode.FileSystemError.NoPermissions('Virtual directory contents are read-only');
    }
    delete() {
        throw vscode.FileSystemError.NoPermissions('Virtual directory contents are read-only');
    }
    rename() {
        throw vscode.FileSystemError.NoPermissions('Virtual directory contents are read-only');
    }
    createDirectory() {
        throw vscode.FileSystemError.NoPermissions('Virtual directory contents are read-only');
    }
}
//# sourceMappingURL=extension.js.map