"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.RecafClient = void 0;
const vscode = __importStar(require("vscode"));
const path = __importStar(require("path"));
const child_process_1 = require("child_process");
const node_1 = require("vscode-jsonrpc/node");
// JSON-RPC Request Types
var Requests;
(function (Requests) {
    // Workspace
    Requests.OPEN_JAR = new node_1.RequestType('workspace/open');
    Requests.CLOSE_WORKSPACE = new node_1.RequestType0('workspace/close');
    Requests.GET_CLASSES = new node_1.RequestType0('workspace/classes');
    Requests.GET_CLASS_INFO = new node_1.RequestType('workspace/classInfo');
    Requests.GET_MEMBERS = new node_1.RequestType('workspace/members');
    Requests.GET_FILES = new node_1.RequestType0('workspace/files');
    Requests.GET_FILE_CONTENT = new node_1.RequestType('workspace/fileContent');
    Requests.EXPORT_JAR = new node_1.RequestType('workspace/export');
    // Decompile
    Requests.DECOMPILE = new node_1.RequestType('decompile/decompileClass');
    Requests.LIST_DECOMPILERS = new node_1.RequestType0('decompile/list');
    Requests.GET_PREFERRED_DECOMPILER = new node_1.RequestType0('decompile/preferred');
    Requests.SET_PREFERRED_DECOMPILER = new node_1.RequestType('decompile/setPreferred');
    // Search
    Requests.SEARCH_STRING = new node_1.RequestType('search/string');
    Requests.SEARCH_NUMBER = new node_1.RequestType('search/number');
    Requests.SEARCH_CLASS_REFERENCE = new node_1.RequestType('search/classReference');
    Requests.SEARCH_MEMBER_REFERENCE = new node_1.RequestType('search/memberReference');
    // Hierarchy
    Requests.GET_SUPER_TYPES = new node_1.RequestType('hierarchy/superTypes');
    Requests.GET_SUB_TYPES = new node_1.RequestType('hierarchy/subTypes');
    Requests.GET_CALLERS = new node_1.RequestType('hierarchy/callers');
    Requests.GET_CALLEES = new node_1.RequestType('hierarchy/callees');
    // Mappings
    Requests.GET_MAPPING_FORMATS = new node_1.RequestType0('mappings/formats');
    Requests.IMPORT_MAPPINGS = new node_1.RequestType('mappings/import');
    Requests.APPLY_MAPPINGS = new node_1.RequestType('mappings/apply');
    Requests.EXPORT_MAPPINGS = new node_1.RequestType('mappings/export');
    Requests.RENAME_CLASS = new node_1.RequestType('mappings/renameClass');
    Requests.RENAME_FIELD = new node_1.RequestType('mappings/renameField');
    Requests.RENAME_METHOD = new node_1.RequestType('mappings/renameMethod');
    Requests.GET_CURRENT_MAPPINGS = new node_1.RequestType0('mappings/current');
    // Transform
    Requests.LIST_TRANSFORMERS = new node_1.RequestType0('transform/list');
    Requests.APPLY_TRANSFORMERS = new node_1.RequestType('transform/apply');
    Requests.RECOMPILE = new node_1.RequestType('transform/recompile');
    Requests.DISASSEMBLE = new node_1.RequestType('transform/disassemble');
    Requests.ASSEMBLE = new node_1.RequestType('transform/assemble');
    // Script
    Requests.LIST_SCRIPTS = new node_1.RequestType0('script/list');
    Requests.RUN_SCRIPT = new node_1.RequestType('script/run');
    Requests.COMPILE_SCRIPT = new node_1.RequestType('script/compile');
    Requests.RUN_SCRIPT_FILE = new node_1.RequestType('script/runFile');
    // Ping/Health check
    Requests.PING = new node_1.RequestType0('ping');
})(Requests || (Requests = {}));
class RecafClient {
    serverProcess = null;
    connection = null;
    context;
    outputChannel;
    pingInterval = null;
    reconnectTimeout = null;
    isReconnecting = false;
    lastPingTime = 0;
    pingTimeout = 10000; // 10 seconds timeout for ping
    pingIntervalMs = 30000; // Ping every 30 seconds
    maxReconnectAttempts = 5;
    reconnectAttempts = 0;
    constructor(context) {
        this.context = context;
        this.outputChannel = vscode.window.createOutputChannel('Java Decompiler Server');
    }
    async checkJavaVersion(javaPath) {
        try {
            const versionOutput = (0, child_process_1.execSync)(`"${javaPath}" -version`, {
                encoding: 'utf8',
                stdio: ['ignore', 'pipe', 'pipe'],
                timeout: 5000
            });
            // Parse Java version from output like "openjdk version "21.0.5""
            const versionMatch = versionOutput.match(/version "(\d+)/);
            if (versionMatch) {
                return parseInt(versionMatch[1], 10);
            }
            // Try alternative format
            const altMatch = versionOutput.match(/version (\d+)/);
            if (altMatch) {
                return parseInt(altMatch[1], 10);
            }
            return 0;
        }
        catch (error) {
            this.outputChannel.appendLine(`Warning: Could not check Java version: ${error}`);
            return 0;
        }
    }
    async start() {
        const config = vscode.workspace.getConfiguration('recaf');
        const javaPath = config.get('javaPath') || 'java';
        // Check Java version (server requires Java 22+)
        const javaVersion = await this.checkJavaVersion(javaPath);
        if (javaVersion > 0 && javaVersion < 22) {
            const errorMsg = `Java version ${javaVersion} detected, but Java Decompiler server requires Java 22 or higher. ` +
                `Please install Java 22+ and configure the path in settings (recaf.javaPath), ` +
                `or ensure JAVA_HOME points to Java 22+.`;
            this.outputChannel.appendLine(`ERROR: ${errorMsg}`);
            throw new Error(errorMsg);
        }
        // Find the server JAR
        let serverJar = config.get('serverJar');
        if (!serverJar) {
            // Use bundled JAR from extension
            const extensionPath = this.context.extensionPath;
            const possiblePaths = [
                path.join(extensionPath, 'server', 'recaf-server.jar'),
                path.join(extensionPath, 'Recaf', 'recaf-server', 'build', 'libs', 'recaf-server-4.0.0-SNAPSHOT-all.jar'),
                path.join(extensionPath, '..', 'blackswan-java-decompiler-recaf', 'Recaf', 'recaf-server', 'build', 'libs', 'recaf-server-4.0.0-SNAPSHOT-all.jar'),
            ];
            this.outputChannel.appendLine('Searching for server JAR in:');
            for (const p of possiblePaths) {
                this.outputChannel.appendLine(`  - ${p}`);
                try {
                    await vscode.workspace.fs.stat(vscode.Uri.file(p));
                    serverJar = p;
                    this.outputChannel.appendLine(`Found server JAR at: ${p}`);
                    break;
                }
                catch (err) {
                    // Path doesn't exist, try next
                }
            }
        }
        if (!serverJar) {
            const errorMsg = 'Could not find Java decompiler server JAR. Please configure recaf.serverJar in settings.';
            this.outputChannel.appendLine(`ERROR: ${errorMsg}`);
            throw new Error(errorMsg);
        }
        // Verify the JAR file exists and is readable
        try {
            await vscode.workspace.fs.stat(vscode.Uri.file(serverJar));
        }
        catch (err) {
            const errorMsg = `Server JAR file not found or not accessible: ${serverJar}`;
            this.outputChannel.appendLine(`ERROR: ${errorMsg}`);
            throw new Error(errorMsg);
        }
        this.outputChannel.appendLine(`Starting Java Decompiler server with Java: ${javaPath}${javaVersion > 0 ? ` (version ${javaVersion})` : ''}`);
        this.outputChannel.appendLine(`Server JAR: ${serverJar}`);
        // Start the server process with proper logging setup
        // stdout must be 'pipe' for JSON-RPC, stderr is 'pipe' for logging
        this.serverProcess = (0, child_process_1.spawn)(javaPath, ['-jar', serverJar], {
            stdio: ['pipe', 'pipe', 'pipe'],
            env: {
                ...process.env,
                // Ensure server logs go to stderr
                JAVA_TOOL_OPTIONS: '-Djava.util.logging.config.file='
            }
        });
        if (!this.serverProcess.stdout || !this.serverProcess.stdin || !this.serverProcess.stderr) {
            throw new Error('Failed to start server process - missing stdio streams');
        }
        // Set up stderr logging with proper buffering and line handling
        let stderrBuffer = '';
        let stderrLineBuffer = '';
        this.serverProcess.stderr.setEncoding('utf8');
        this.serverProcess.stderr.on('data', (data) => {
            const output = typeof data === 'string' ? data : data.toString('utf8');
            stderrBuffer += output;
            stderrLineBuffer += output;
            // Process complete lines
            const lines = stderrLineBuffer.split('\n');
            stderrLineBuffer = lines.pop() || ''; // Keep incomplete line in buffer
            for (const line of lines) {
                if (line.trim().length > 0) {
                    this.outputChannel.appendLine(`[Server] ${line.trim()}`);
                }
            }
            // Check for Java version errors
            if (output.includes('UnsupportedClassVersionError') || output.includes('class file version')) {
                const versionMatch = output.match(/class file version (\d+\.\d+)/);
                if (versionMatch) {
                    const requiredVersion = Math.floor(parseFloat(versionMatch[1]) - 44); // Convert class file version to Java version
                    const errorMsg = `Java version mismatch: Server requires Java ${requiredVersion}+, but current Java version is ${javaVersion > 0 ? javaVersion : 'unknown'}. ` +
                        `Please install Java ${requiredVersion}+ and configure recaf.javaPath in settings.`;
                    this.outputChannel.appendLine(`ERROR: ${errorMsg}`);
                    vscode.window.showErrorMessage(errorMsg);
                }
            }
            // Check for JAR file access errors
            if (output.includes('Unable to access jarfile') || output.includes('Error: Unable to access jarfile')) {
                const errorMsg = `Cannot access server JAR file: ${serverJar}\n` +
                    `Please verify the file exists and is readable.`;
                this.outputChannel.appendLine(`ERROR: ${errorMsg}`);
                vscode.window.showErrorMessage(errorMsg);
            }
        });
        this.serverProcess.stderr.on('end', () => {
            // Process any remaining buffered data
            if (stderrLineBuffer.trim().length > 0) {
                this.outputChannel.appendLine(`[Server] ${stderrLineBuffer.trim()}`);
            }
        });
        this.serverProcess.on('error', (error) => {
            this.outputChannel.appendLine(`Server error: ${error.message}`);
            let errorMsg = `Java decompiler server error: ${error.message}`;
            // Provide helpful error messages
            if (error.message.includes('ENOENT') || error.message.includes('spawn')) {
                errorMsg = `Could not find Java executable at "${javaPath}". ` +
                    `Please ensure Java is installed and configure the correct path in settings (recaf.javaPath).`;
            }
            vscode.window.showErrorMessage(errorMsg);
        });
        this.serverProcess.on('exit', (code) => {
            this.outputChannel.appendLine(`Server exited with code ${code}`);
            if (code !== 0 && code !== null) {
                // Check if it was a version error
                if (stderrBuffer.includes('UnsupportedClassVersionError')) {
                    // Error already shown above
                    return;
                }
                vscode.window.showErrorMessage(`Java Decompiler server exited with code ${code}. Check the output channel for details.`);
            }
        });
        // Create JSON-RPC connection
        // Note: stdout is consumed by JSON-RPC reader, so server logs should go to stderr
        this.connection = (0, node_1.createMessageConnection)(new node_1.StreamMessageReader(this.serverProcess.stdout), new node_1.StreamMessageWriter(this.serverProcess.stdin));
        this.connection.listen();
        this.outputChannel.appendLine('JSON-RPC connection established');
        // Set up error handler for connection
        this.connection.onError(([error, message, count]) => {
            const errorMsg = error instanceof Error ? error.message : String(error);
            this.outputChannel.appendLine(`[Connection Error] ${errorMsg}${message ? ` (message: ${JSON.stringify(message)})` : ''}${count !== undefined ? ` (count: ${count})` : ''}`);
        });
        this.connection.onClose(() => {
            this.outputChannel.appendLine('[Connection] Server connection closed');
            this.stopPingInterval();
            this.handleConnectionLoss();
        });
        // Wait for server to be ready by attempting a simple request
        // This ensures the server is fully initialized before we consider it ready
        await this.waitForServerReady();
        // Start ping interval to monitor connection health
        this.startPingInterval();
    }
    startPingInterval() {
        this.stopPingInterval();
        this.outputChannel.appendLine(`Starting ping interval (every ${this.pingIntervalMs}ms)`);
        this.pingInterval = setInterval(async () => {
            await this.ping();
        }, this.pingIntervalMs);
    }
    stopPingInterval() {
        if (this.pingInterval) {
            clearInterval(this.pingInterval);
            this.pingInterval = null;
        }
    }
    async ping() {
        try {
            const connection = this.ensureConnection();
            this.lastPingTime = Date.now();
            // Try PING first, fall back to LIST_DECOMPILERS if PING is not supported
            let result;
            try {
                result = await Promise.race([
                    connection.sendRequest(Requests.PING),
                    new Promise((_, reject) => setTimeout(() => reject(new Error('Ping timeout')), this.pingTimeout))
                ]);
            }
            catch (pingError) {
                // If PING fails with method not found, try LIST_DECOMPILERS as fallback
                const pingErrorMsg = pingError instanceof Error ? pingError.message : String(pingError);
                if (pingErrorMsg.includes('method not found') || pingErrorMsg.includes('Unknown method')) {
                    this.outputChannel.appendLine('[Ping] PING method not supported, using LIST_DECOMPILERS as health check');
                    result = await Promise.race([
                        connection.sendRequest(Requests.LIST_DECOMPILERS),
                        new Promise((_, reject) => setTimeout(() => reject(new Error('Health check timeout')), this.pingTimeout))
                    ]);
                }
                else {
                    throw pingError;
                }
            }
            const responseTime = Date.now() - this.lastPingTime;
            this.outputChannel.appendLine(`[Ping] Server responded in ${responseTime}ms`);
            this.reconnectAttempts = 0; // Reset reconnect attempts on successful ping
        }
        catch (error) {
            const errorMsg = error instanceof Error ? error.message : String(error);
            this.outputChannel.appendLine(`[Ping] Failed: ${errorMsg}`);
            // If ping fails, try to reconnect
            if (!this.isReconnecting) {
                this.handleConnectionLoss();
            }
        }
    }
    handleConnectionLoss() {
        if (this.isReconnecting) {
            return; // Already handling reconnection
        }
        this.reconnectAttempts++;
        if (this.reconnectAttempts > this.maxReconnectAttempts) {
            this.outputChannel.appendLine(`[Reconnect] Max reconnect attempts (${this.maxReconnectAttempts}) reached. Stopping reconnection.`);
            vscode.window.showErrorMessage('Java Decompiler server connection lost. Please restart the extension.');
            return;
        }
        this.isReconnecting = true;
        const delay = Math.min(1000 * Math.pow(2, this.reconnectAttempts - 1), 30000); // Exponential backoff, max 30s
        this.outputChannel.appendLine(`[Reconnect] Attempting to reconnect in ${delay}ms (attempt ${this.reconnectAttempts}/${this.maxReconnectAttempts})...`);
        this.reconnectTimeout = setTimeout(async () => {
            try {
                // Check if server process is still alive
                if (this.serverProcess && !this.serverProcess.killed) {
                    this.outputChannel.appendLine('[Reconnect] Server process is still running, reinitializing connection...');
                    // Reinitialize connection
                    await this.reinitializeConnection();
                }
                else {
                    this.outputChannel.appendLine('[Reconnect] Server process is not running, attempting restart...');
                    // Restart server
                    await this.restart();
                }
                this.isReconnecting = false;
            }
            catch (error) {
                this.outputChannel.appendLine(`[Reconnect] Failed: ${error}`);
                this.isReconnecting = false;
                // Schedule another reconnect attempt
                this.handleConnectionLoss();
            }
        }, delay);
    }
    async reinitializeConnection() {
        if (!this.serverProcess || !this.serverProcess.stdout || !this.serverProcess.stdin) {
            throw new Error('Cannot reinitialize: server process or streams not available');
        }
        // Dispose old connection
        if (this.connection) {
            this.connection.dispose();
        }
        // Create new connection
        this.connection = (0, node_1.createMessageConnection)(new node_1.StreamMessageReader(this.serverProcess.stdout), new node_1.StreamMessageWriter(this.serverProcess.stdin));
        this.connection.listen();
        this.outputChannel.appendLine('[Reconnect] Connection reinitialized');
        // Set up error handlers
        this.connection.onError(([error, message, count]) => {
            const errorMsg = error instanceof Error ? error.message : String(error);
            this.outputChannel.appendLine(`[Connection Error] ${errorMsg}${message ? ` (message: ${JSON.stringify(message)})` : ''}${count !== undefined ? ` (count: ${count})` : ''}`);
        });
        this.connection.onClose(() => {
            this.outputChannel.appendLine('[Connection] Server connection closed');
            this.stopPingInterval();
            this.handleConnectionLoss();
        });
        // Verify connection works
        await this.waitForServerReady(5, 1000);
        this.startPingInterval();
        this.outputChannel.appendLine('[Reconnect] Successfully reconnected');
    }
    async restart() {
        this.outputChannel.appendLine('[Restart] Stopping server...');
        this.stop();
        // Wait a bit before restarting
        await new Promise(resolve => setTimeout(resolve, 2000));
        this.outputChannel.appendLine('[Restart] Starting server...');
        await this.start();
    }
    async waitForServerReady(maxAttempts = 20, delayMs = 1000) {
        this.outputChannel.appendLine('Waiting for server to initialize...');
        for (let i = 0; i < maxAttempts; i++) {
            try {
                // Try to get decompilers list as a readiness check
                const connection = this.ensureConnection();
                this.outputChannel.appendLine(`Attempting readiness check (${i + 1}/${maxAttempts})...`);
                // Increase timeout for initial requests - server might be loading classes
                const timeout = i < 5 ? 5000 : 3000; // Longer timeout for first few attempts
                await Promise.race([
                    connection.sendRequest(Requests.LIST_DECOMPILERS),
                    new Promise((_, reject) => setTimeout(() => reject(new Error(`Request timeout after ${timeout}ms`)), timeout))
                ]);
                this.outputChannel.appendLine('✓ Server is ready and responding');
                return;
            }
            catch (error) {
                const errorMsg = error instanceof Error ? error.message : String(error);
                if (i < maxAttempts - 1) {
                    if (errorMsg.includes('timeout') || errorMsg.includes('Timeout')) {
                        this.outputChannel.appendLine(`Server not responding yet (${errorMsg}), waiting ${delayMs}ms...`);
                    }
                    else {
                        this.outputChannel.appendLine(`Readiness check failed (${errorMsg}), retrying...`);
                    }
                    await new Promise(resolve => setTimeout(resolve, delayMs));
                }
                else {
                    this.outputChannel.appendLine(`⚠ Warning: Server readiness check failed after ${maxAttempts} attempts: ${errorMsg}`);
                    this.outputChannel.appendLine('⚠ Continuing anyway - server may still be initializing. Some operations may fail.');
                    // Don't throw - allow the extension to continue even if readiness check fails
                }
            }
        }
    }
    stop() {
        this.stopPingInterval();
        if (this.reconnectTimeout) {
            clearTimeout(this.reconnectTimeout);
            this.reconnectTimeout = null;
        }
        this.isReconnecting = false;
        this.reconnectAttempts = 0;
        if (this.connection) {
            this.connection.dispose();
            this.connection = null;
        }
        if (this.serverProcess) {
            this.serverProcess.kill();
            this.serverProcess = null;
        }
        this.outputChannel.appendLine('Java decompiler server stopped');
    }
    ensureConnection() {
        if (!this.connection) {
            throw new Error('Java decompiler server is not connected');
        }
        // Check if connection is still alive by checking if process is running
        if (this.serverProcess && this.serverProcess.killed) {
            throw new Error('Server process has been killed');
        }
        return this.connection;
    }
    // Public method to check if server is connected and healthy
    async isConnected() {
        try {
            if (!this.connection) {
                return false;
            }
            const connection = this.ensureConnection();
            // Try PING, fall back to LIST_DECOMPILERS
            try {
                await connection.sendRequest(Requests.PING);
            }
            catch {
                await connection.sendRequest(Requests.LIST_DECOMPILERS);
            }
            return true;
        }
        catch {
            return false;
        }
    }
    // Workspace operations
    async openJar(filePath) {
        return this.ensureConnection().sendRequest(Requests.OPEN_JAR, { path: filePath });
    }
    async closeWorkspace() {
        return this.ensureConnection().sendRequest(Requests.CLOSE_WORKSPACE);
    }
    async getClasses() {
        return this.ensureConnection().sendRequest(Requests.GET_CLASSES);
    }
    async getClassInfo(name) {
        return this.ensureConnection().sendRequest(Requests.GET_CLASS_INFO, { name });
    }
    async getMembers(className) {
        return this.ensureConnection().sendRequest(Requests.GET_MEMBERS, { className });
    }
    async getFiles() {
        return this.ensureConnection().sendRequest(Requests.GET_FILES);
    }
    async getFileContent(name) {
        return this.ensureConnection().sendRequest(Requests.GET_FILE_CONTENT, { name });
    }
    async exportJar(filePath) {
        return this.ensureConnection().sendRequest(Requests.EXPORT_JAR, { path: filePath });
    }
    // Decompile operations
    async decompile(className, decompiler) {
        const response = await this.ensureConnection().sendRequest(Requests.DECOMPILE, { className, decompiler });
        // Map server response to client interface
        return {
            source: response.source || '',
            decompiler: response.decompiler || 'unknown',
            success: response.source != null && response.error == null,
            error: response.error || undefined
        };
    }
    async listDecompilers() {
        return this.ensureConnection().sendRequest(Requests.LIST_DECOMPILERS);
    }
    async getPreferredDecompiler() {
        return this.ensureConnection().sendRequest(Requests.GET_PREFERRED_DECOMPILER);
    }
    async setPreferredDecompiler(name) {
        return this.ensureConnection().sendRequest(Requests.SET_PREFERRED_DECOMPILER, { name });
    }
    // Search operations
    async searchString(query, mode) {
        return this.ensureConnection().sendRequest(Requests.SEARCH_STRING, { query, mode });
    }
    async searchNumber(value) {
        return this.ensureConnection().sendRequest(Requests.SEARCH_NUMBER, { value });
    }
    async searchClassReference(owner, mode) {
        return this.ensureConnection().sendRequest(Requests.SEARCH_CLASS_REFERENCE, { owner, mode });
    }
    async searchMemberReference(owner, name, descriptor, mode) {
        return this.ensureConnection().sendRequest(Requests.SEARCH_MEMBER_REFERENCE, { owner, name, descriptor, mode });
    }
    // Hierarchy operations
    async getSuperTypes(className) {
        return this.ensureConnection().sendRequest(Requests.GET_SUPER_TYPES, { className });
    }
    async getSubTypes(className) {
        return this.ensureConnection().sendRequest(Requests.GET_SUB_TYPES, { className });
    }
    async getCallers(owner, name, desc) {
        return this.ensureConnection().sendRequest(Requests.GET_CALLERS, { owner, name, desc });
    }
    async getCallees(owner, name, desc) {
        return this.ensureConnection().sendRequest(Requests.GET_CALLEES, { owner, name, desc });
    }
    // Mapping operations
    async getMappingFormats() {
        return this.ensureConnection().sendRequest(Requests.GET_MAPPING_FORMATS);
    }
    async importMappings(filePath, format) {
        return this.ensureConnection().sendRequest(Requests.IMPORT_MAPPINGS, { filePath, format });
    }
    async applyMappings(filePath, format) {
        return this.ensureConnection().sendRequest(Requests.APPLY_MAPPINGS, { filePath, format });
    }
    async exportMappings(filePath, format) {
        return this.ensureConnection().sendRequest(Requests.EXPORT_MAPPINGS, { filePath, format });
    }
    async renameClass(oldName, newName) {
        return this.ensureConnection().sendRequest(Requests.RENAME_CLASS, { oldName, newName });
    }
    async renameField(ownerName, fieldName, fieldDesc, newName) {
        return this.ensureConnection().sendRequest(Requests.RENAME_FIELD, { ownerName, fieldName, fieldDesc, newName });
    }
    async renameMethod(ownerName, methodName, methodDesc, newName) {
        return this.ensureConnection().sendRequest(Requests.RENAME_METHOD, { ownerName, methodName, methodDesc, newName });
    }
    async getCurrentMappings() {
        return this.ensureConnection().sendRequest(Requests.GET_CURRENT_MAPPINGS);
    }
    // Transform operations
    async listTransformers() {
        return this.ensureConnection().sendRequest(Requests.LIST_TRANSFORMERS);
    }
    async applyTransformers(transformerNames) {
        return this.ensureConnection().sendRequest(Requests.APPLY_TRANSFORMERS, { transformerNames });
    }
    async recompile(className, source) {
        return this.ensureConnection().sendRequest(Requests.RECOMPILE, { className, source });
    }
    async disassemble(className) {
        return this.ensureConnection().sendRequest(Requests.DISASSEMBLE, { className });
    }
    async assemble(className, source) {
        return this.ensureConnection().sendRequest(Requests.ASSEMBLE, { className, source });
    }
    // Script operations
    async listScripts() {
        return this.ensureConnection().sendRequest(Requests.LIST_SCRIPTS);
    }
    async runScript(source) {
        return this.ensureConnection().sendRequest(Requests.RUN_SCRIPT, { source });
    }
    async compileScript(source) {
        return this.ensureConnection().sendRequest(Requests.COMPILE_SCRIPT, { source });
    }
    async runScriptFile(fileName) {
        return this.ensureConnection().sendRequest(Requests.RUN_SCRIPT_FILE, { fileName });
    }
}
exports.RecafClient = RecafClient;
//# sourceMappingURL=RecafClient.js.map